<?php
/**
 * Admin Dashboard
 */

session_start();
require_once '../config/config.php';
require_once '../config/database.php';

// Check authentication
if (!isset($_SESSION['admin_id'])) {
    header('Location: index.php');
    exit();
}

try {
    $db = new Database();
    $conn = $db->getConnection();
    
    // Get statistics
    $stats = [];
    
    // Total users
    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM users WHERE is_active = 1");
    $stmt->execute();
    $stats['total_users'] = $stmt->fetch()['count'];
    
    // Total games
    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM games WHERE status = 'completed'");
    $stmt->execute();
    $stats['total_games'] = $stmt->fetch()['count'];
    
    // Active games
    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM games WHERE status = 'active'");
    $stmt->execute();
    $stats['active_games'] = $stmt->fetch()['count'];
    
    // Total transactions
    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM transactions WHERE status = 'completed'");
    $stmt->execute();
    $stats['total_transactions'] = $stmt->fetch()['count'];
    
    // Total deposits
    $stmt = $conn->prepare("
        SELECT SUM(amount) as total FROM transactions 
        WHERE type = 'deposit' AND status = 'completed'
    ");
    $stmt->execute();
    $stats['total_deposits'] = $stmt->fetch()['total'] ?? 0;
    
    // Total withdrawals
    $stmt = $conn->prepare("
        SELECT SUM(amount) as total FROM transactions 
        WHERE type = 'withdrawal' AND status = 'completed'
    ");
    $stmt->execute();
    $stats['total_withdrawals'] = $stmt->fetch()['total'] ?? 0;
    
    // Pending withdrawals
    $stmt = $conn->prepare("
        SELECT COUNT(*) as count FROM withdrawals WHERE status = 'pending'
    ");
    $stmt->execute();
    $stats['pending_withdrawals'] = $stmt->fetch()['count'];
    
    // Recent users
    $stmt = $conn->prepare("
        SELECT username, email, created_at 
        FROM users 
        WHERE is_active = 1 
        ORDER BY created_at DESC 
        LIMIT 5
    ");
    $stmt->execute();
    $recent_users = $stmt->fetchAll();
    
    // Recent games
    $stmt = $conn->prepare("
        SELECT g.*, u1.username as player1_username, u2.username as player2_username
        FROM games g
        LEFT JOIN users u1 ON g.player1_id = u1.id
        LEFT JOIN users u2 ON g.player2_id = u2.id
        ORDER BY g.created_at DESC 
        LIMIT 5
    ");
    $stmt->execute();
    $recent_games = $stmt->fetchAll();
    
} catch (Exception $e) {
    $error = 'Failed to load dashboard data';
    $stats = [];
    $recent_users = [];
    $recent_games = [];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard - Checkers Game</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: #f8f9fa;
            color: #333;
        }
        
        .header {
            background: #007AFF;
            color: white;
            padding: 1rem 2rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .header h1 {
            font-size: 1.5rem;
        }
        
        .header-actions {
            display: flex;
            gap: 1rem;
            align-items: center;
        }
        
        .btn {
            padding: 0.5rem 1rem;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            text-decoration: none;
            display: inline-block;
            font-size: 0.9rem;
        }
        
        .btn-primary {
            background: #28a745;
            color: white;
        }
        
        .btn-secondary {
            background: #6c757d;
            color: white;
        }
        
        .btn-danger {
            background: #dc3545;
            color: white;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 2rem;
        }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }
        
        .stat-card {
            background: white;
            padding: 1.5rem;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        
        .stat-title {
            font-size: 0.9rem;
            color: #666;
            margin-bottom: 0.5rem;
        }
        
        .stat-value {
            font-size: 2rem;
            font-weight: bold;
            color: #333;
        }
        
        .content-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 2rem;
        }
        
        .content-card {
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        
        .card-header {
            background: #f8f9fa;
            padding: 1rem;
            border-bottom: 1px solid #e9ecef;
        }
        
        .card-title {
            font-size: 1.1rem;
            font-weight: bold;
            color: #333;
        }
        
        .card-body {
            padding: 1rem;
        }
        
        .table {
            width: 100%;
            border-collapse: collapse;
        }
        
        .table th,
        .table td {
            padding: 0.75rem;
            text-align: left;
            border-bottom: 1px solid #e9ecef;
        }
        
        .table th {
            background: #f8f9fa;
            font-weight: 600;
            color: #333;
        }
        
        .table tbody tr:hover {
            background: #f8f9fa;
        }
        
        .status-badge {
            padding: 0.25rem 0.5rem;
            border-radius: 3px;
            font-size: 0.8rem;
            font-weight: 500;
        }
        
        .status-active {
            background: #d4edda;
            color: #155724;
        }
        
        .status-completed {
            background: #d1ecf1;
            color: #0c5460;
        }
        
        .status-pending {
            background: #fff3cd;
            color: #856404;
        }
        
        .error {
            background: #f8d7da;
            color: #721c24;
            padding: 1rem;
            border-radius: 5px;
            margin-bottom: 1rem;
        }
        
        @media (max-width: 768px) {
            .content-grid {
                grid-template-columns: 1fr;
            }
            
            .stats-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <div class="header">
        <h1>Admin Dashboard</h1>
        <div class="header-actions">
            <span>Welcome, <?php echo htmlspecialchars($_SESSION['admin_username']); ?></span>
            <a href="users.php" class="btn btn-primary">Manage Users</a>
            <a href="competitions.php" class="btn btn-primary">Competitions</a>
            <a href="withdrawals.php" class="btn btn-secondary">Withdrawals</a>
            <a href="logout.php" class="btn btn-danger">Logout</a>
        </div>
    </div>
    
    <div class="container">
        <?php if (isset($error)): ?>
            <div class="error"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>
        
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-title">Total Users</div>
                <div class="stat-value"><?php echo number_format($stats['total_users'] ?? 0); ?></div>
            </div>
            
            <div class="stat-card">
                <div class="stat-title">Total Games</div>
                <div class="stat-value"><?php echo number_format($stats['total_games'] ?? 0); ?></div>
            </div>
            
            <div class="stat-card">
                <div class="stat-title">Active Games</div>
                <div class="stat-value"><?php echo number_format($stats['active_games'] ?? 0); ?></div>
            </div>
            
            <div class="stat-card">
                <div class="stat-title">Total Deposits</div>
                <div class="stat-value">TZS <?php echo number_format($stats['total_deposits'] ?? 0); ?></div>
            </div>
            
            <div class="stat-card">
                <div class="stat-title">Total Withdrawals</div>
                <div class="stat-value">TZS <?php echo number_format($stats['total_withdrawals'] ?? 0); ?></div>
            </div>
            
            <div class="stat-card">
                <div class="stat-title">Pending Withdrawals</div>
                <div class="stat-value"><?php echo number_format($stats['pending_withdrawals'] ?? 0); ?></div>
            </div>
        </div>
        
        <div class="content-grid">
            <div class="content-card">
                <div class="card-header">
                    <h2 class="card-title">Recent Users</h2>
                </div>
                <div class="card-body">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Username</th>
                                <th>Email</th>
                                <th>Joined</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($recent_users as $user): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($user['username']); ?></td>
                                    <td><?php echo htmlspecialchars($user['email']); ?></td>
                                    <td><?php echo date('M j, Y', strtotime($user['created_at'])); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            
            <div class="content-card">
                <div class="card-header">
                    <h2 class="card-title">Recent Games</h2>
                </div>
                <div class="card-body">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Players</th>
                                <th>Stake</th>
                                <th>Status</th>
                                <th>Date</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($recent_games as $game): ?>
                                <tr>
                                    <td>
                                        <?php echo htmlspecialchars($game['player1_username']); ?> vs 
                                        <?php echo htmlspecialchars($game['player2_username'] ?? 'Bot'); ?>
                                    </td>
                                    <td>TZS <?php echo number_format($game['stake_amount']); ?></td>
                                    <td>
                                        <span class="status-badge status-<?php echo $game['status']; ?>">
                                            <?php echo ucfirst($game['status']); ?>
                                        </span>
                                    </td>
                                    <td><?php echo date('M j, H:i', strtotime($game['created_at'])); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</body>
</html>
