<?php
/**
 * Giveaway Checkers Game Engine
 * Implements the complete game logic for Giveaway Checkers
 */

class CheckersEngine {
    private $board;
    private $currentPlayer;
    private $gameOver;
    private $winner;
    
    public function __construct() {
        $this->initializeBoard();
        $this->currentPlayer = 1; // Player 1 (darker pieces) moves first
        $this->gameOver = false;
        $this->winner = null;
    }
    
    private function initializeBoard() {
        // 8x8 board, 0 = empty, 1 = player1, 2 = player2, 3 = player1 king, 4 = player2 king
        $this->board = [
            [0, 1, 0, 1, 0, 1, 0, 1],
            [1, 0, 1, 0, 1, 0, 1, 0],
            [0, 1, 0, 1, 0, 1, 0, 1],
            [0, 0, 0, 0, 0, 0, 0, 0],
            [0, 0, 0, 0, 0, 0, 0, 0],
            [2, 0, 2, 0, 2, 0, 2, 0],
            [0, 2, 0, 2, 0, 2, 0, 2],
            [2, 0, 2, 0, 2, 0, 2, 0]
        ];
    }
    
    public function getBoard() {
        return $this->board;
    }
    
    public function getCurrentPlayer() {
        return $this->currentPlayer;
    }
    
    public function isGameOver() {
        return $this->gameOver;
    }
    
    public function getWinner() {
        return $this->winner;
    }
    
    public function makeMove($from, $to) {
        if ($this->gameOver) {
            return ['valid' => false, 'message' => 'Game is over'];
        }
        
        $fromPos = $this->parsePosition($from);
        $toPos = $this->parsePosition($to);
        
        if (!$fromPos || !$toPos) {
            return ['valid' => false, 'message' => 'Invalid position format'];
        }
        
        // Check if it's the correct player's turn
        $piece = $this->board[$fromPos['row']][$fromPos['col']];
        if (!$this->isPlayerPiece($piece, $this->currentPlayer)) {
            return ['valid' => false, 'message' => 'Not your piece'];
        }
        
        // Check if move is valid
        $moveResult = $this->validateMove($fromPos, $toPos);
        if (!$moveResult['valid']) {
            return $moveResult;
        }
        
        // Execute the move
        $this->executeMove($fromPos, $toPos, $moveResult['captures']);
        
        // Check for kinging
        $this->checkForKinging($toPos);
        
        // Check if game is over
        $this->checkGameOver();
        
        // Switch players if game continues
        if (!$this->gameOver) {
            $this->currentPlayer = $this->currentPlayer === 1 ? 2 : 1;
        }
        
        return ['valid' => true, 'message' => 'Move successful'];
    }
    
    private function parsePosition($position) {
        if (strlen($position) !== 2) {
            return false;
        }
        
        $col = ord(strtolower($position[0])) - ord('a');
        $row = intval($position[1]) - 1;
        
        if ($col < 0 || $col > 7 || $row < 0 || $row > 7) {
            return false;
        }
        
        return ['row' => $row, 'col' => $col];
    }
    
    private function isPlayerPiece($piece, $player) {
        if ($player === 1) {
            return $piece === 1 || $piece === 3; // Player 1 or Player 1 King
        } else {
            return $piece === 2 || $piece === 4; // Player 2 or Player 2 King
        }
    }
    
    private function validateMove($from, $to) {
        $piece = $this->board[$from['row']][$from['col']];
        $targetPiece = $this->board[$to['row']][$to['col']];
        
        // Target must be empty
        if ($targetPiece !== 0) {
            return ['valid' => false, 'message' => 'Target position is occupied'];
        }
        
        // Check if any captures are available (mandatory)
        $allCaptures = $this->findAllCaptures($this->currentPlayer);
        if (!empty($allCaptures)) {
            // Must make a capture if available
            $isCaptureMove = $this->isCaptureMove($from, $to);
            if (!$isCaptureMove) {
                return ['valid' => false, 'message' => 'You must capture if possible'];
            }
        }
        
        // Check if move is valid for the piece type
        if ($piece === 1 || $piece === 2) {
            // Regular piece
            return $this->validateRegularMove($from, $to, $piece);
        } else {
            // King piece
            return $this->validateKingMove($from, $to, $piece);
        }
    }
    
    private function validateRegularMove($from, $to, $piece) {
        $rowDiff = $to['row'] - $from['row'];
        $colDiff = $to['col'] - $from['col'];
        
        // Must move diagonally
        if (abs($rowDiff) !== abs($colDiff)) {
            return ['valid' => false, 'message' => 'Must move diagonally'];
        }
        
        // Regular pieces can only move forward
        if ($piece === 1 && $rowDiff <= 0) { // Player 1 moves down
            return ['valid' => false, 'message' => 'Regular pieces can only move forward'];
        }
        if ($piece === 2 && $rowDiff >= 0) { // Player 2 moves up
            return ['valid' => false, 'message' => 'Regular pieces can only move forward'];
        }
        
        // Check distance
        if (abs($rowDiff) > 2) {
            return ['valid' => false, 'message' => 'Invalid move distance'];
        }
        
        // Handle captures
        if (abs($rowDiff) === 2) {
            return $this->validateCapture($from, $to);
        }
        
        // Simple move
        if (abs($rowDiff) === 1) {
            return ['valid' => true, 'captures' => []];
        }
        
        return ['valid' => false, 'message' => 'Invalid move'];
    }
    
    private function validateKingMove($from, $to, $piece) {
        $rowDiff = $to['row'] - $from['row'];
        $colDiff = $to['col'] - $from['col'];
        
        // Must move diagonally
        if (abs($rowDiff) !== abs($colDiff)) {
            return ['valid' => false, 'message' => 'Must move diagonally'];
        }
        
        // Check distance
        if (abs($rowDiff) > 2) {
            return ['valid' => false, 'message' => 'Invalid move distance'];
        }
        
        // Handle captures
        if (abs($rowDiff) === 2) {
            return $this->validateCapture($from, $to);
        }
        
        // Simple move
        if (abs($rowDiff) === 1) {
            return ['valid' => true, 'captures' => []];
        }
        
        return ['valid' => false, 'message' => 'Invalid move'];
    }
    
    private function validateCapture($from, $to) {
        $middleRow = ($from['row'] + $to['row']) / 2;
        $middleCol = ($from['col'] + $to['col']) / 2;
        
        $middlePiece = $this->board[$middleRow][$middleCol];
        
        // Middle piece must be opponent's piece
        if ($middlePiece === 0) {
            return ['valid' => false, 'message' => 'No piece to capture'];
        }
        
        if ($this->isPlayerPiece($middlePiece, $this->currentPlayer)) {
            return ['valid' => false, 'message' => 'Cannot capture your own piece'];
        }
        
        return [
            'valid' => true,
            'captures' => [['row' => $middleRow, 'col' => $middleCol]]
        ];
    }
    
    private function isCaptureMove($from, $to) {
        $rowDiff = abs($to['row'] - $from['row']);
        $colDiff = abs($to['col'] - $from['col']);
        
        if ($rowDiff === 2 && $colDiff === 2) {
            $middleRow = ($from['row'] + $to['row']) / 2;
            $middleCol = ($from['col'] + $to['col']) / 2;
            $middlePiece = $this->board[$middleRow][$middleCol];
            
            return $middlePiece !== 0 && !$this->isPlayerPiece($middlePiece, $this->currentPlayer);
        }
        
        return false;
    }
    
    private function findAllCaptures($player) {
        $captures = [];
        
        for ($row = 0; $row < 8; $row++) {
            for ($col = 0; $col < 8; $col++) {
                $piece = $this->board[$row][$col];
                if ($this->isPlayerPiece($piece, $player)) {
                    $pieceCaptures = $this->findCapturesForPiece($row, $col, $piece);
                    $captures = array_merge($captures, $pieceCaptures);
                }
            }
        }
        
        return $captures;
    }
    
    private function findCapturesForPiece($row, $col, $piece) {
        $captures = [];
        $directions = [];
        
        if ($piece === 1) { // Player 1 regular piece
            $directions = [[1, 1], [1, -1]];
        } elseif ($piece === 2) { // Player 2 regular piece
            $directions = [[-1, 1], [-1, -1]];
        } else { // King pieces
            $directions = [[1, 1], [1, -1], [-1, 1], [-1, -1]];
        }
        
        foreach ($directions as $dir) {
            $newRow = $row + $dir[0];
            $newCol = $col + $dir[1];
            $jumpRow = $row + 2 * $dir[0];
            $jumpCol = $col + 2 * $dir[1];
            
            if ($jumpRow >= 0 && $jumpRow < 8 && $jumpCol >= 0 && $jumpCol < 8) {
                $middlePiece = $this->board[$newRow][$newCol];
                $targetPiece = $this->board[$jumpRow][$jumpCol];
                
                if ($middlePiece !== 0 && 
                    !$this->isPlayerPiece($middlePiece, $this->currentPlayer) && 
                    $targetPiece === 0) {
                    $captures[] = [
                        'from' => ['row' => $row, 'col' => $col],
                        'to' => ['row' => $jumpRow, 'col' => $jumpCol],
                        'capture' => ['row' => $newRow, 'col' => $newCol]
                    ];
                }
            }
        }
        
        return $captures;
    }
    
    private function executeMove($from, $to, $captures) {
        $piece = $this->board[$from['row']][$from['col']];
        
        // Move the piece
        $this->board[$to['row']][$to['col']] = $piece;
        $this->board[$from['row']][$from['col']] = 0;
        
        // Remove captured pieces
        foreach ($captures as $capture) {
            $this->board[$capture['row']][$capture['col']] = 0;
        }
    }
    
    private function checkForKinging($position) {
        $piece = $this->board[$position['row']][$position['col']];
        
        if ($piece === 1 && $position['row'] === 7) {
            $this->board[$position['row']][$position['col']] = 3; // King
        } elseif ($piece === 2 && $position['row'] === 0) {
            $this->board[$position['row']][$position['col']] = 4; // King
        }
    }
    
    private function checkGameOver() {
        // Check if current player has any legal moves
        $hasMoves = $this->hasLegalMoves($this->currentPlayer);
        
        if (!$hasMoves) {
            $this->gameOver = true;
            // In Giveaway Checkers, the player who cannot move WINS
            $this->winner = $this->currentPlayer;
        }
    }
    
    private function hasLegalMoves($player) {
        // Check all pieces for the player
        for ($row = 0; $row < 8; $row++) {
            for ($col = 0; $col < 8; $col++) {
                $piece = $this->board[$row][$col];
                if ($this->isPlayerPiece($piece, $player)) {
                    if ($this->hasMovesForPiece($row, $col, $piece)) {
                        return true;
                    }
                }
            }
        }
        
        return false;
    }
    
    private function hasMovesForPiece($row, $col, $piece) {
        $directions = [];
        
        if ($piece === 1) { // Player 1 regular piece
            $directions = [[1, 1], [1, -1]];
        } elseif ($piece === 2) { // Player 2 regular piece
            $directions = [[-1, 1], [-1, -1]];
        } else { // King pieces
            $directions = [[1, 1], [1, -1], [-1, 1], [-1, -1]];
        }
        
        foreach ($directions as $dir) {
            $newRow = $row + $dir[0];
            $newCol = $col + $dir[1];
            
            if ($newRow >= 0 && $newRow < 8 && $newCol >= 0 && $newCol < 8) {
                if ($this->board[$newRow][$newCol] === 0) {
                    return true;
                }
            }
        }
        
        return false;
    }
    
    public function getGameState() {
        return [
            'board' => $this->board,
            'currentPlayer' => $this->currentPlayer,
            'gameOver' => $this->gameOver,
            'winner' => $this->winner
        ];
    }
    
    public function loadGameState($board, $currentPlayer, $gameOver = false, $winner = null) {
        $this->board = $board;
        $this->currentPlayer = $currentPlayer;
        $this->gameOver = $gameOver;
        $this->winner = $winner;
    }
}
?>
