<?php
/**
 * Create Game Room
 * POST /api/rooms/create
 */

require_once '../../config/config.php';
require_once '../../config/database.php';
require_once '../../config/jwt.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    error('Method not allowed', 'METHOD_NOT_ALLOWED', 405);
}

try {
    $user_id = getCurrentUser();
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        error('Invalid JSON input', 'INVALID_INPUT');
    }
    
    // Validate required fields
    validateRequired($input, ['stake_amount']);
    
    $stake_amount = floatval($input['stake_amount']);
    
    // Validate stake amount
    if ($stake_amount < 100) {
        error('Minimum stake amount is 100 TZS', 'INVALID_STAKE');
    }
    
    if ($stake_amount > 100000) {
        error('Maximum stake amount is 100,000 TZS', 'INVALID_STAKE');
    }
    
    $db = new Database();
    $conn = $db->getConnection();
    
    // Check user's balance
    $stmt = $conn->prepare("SELECT main_balance, bonus_balance FROM wallets WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $wallet = $stmt->fetch();
    
    if (!$wallet) {
        error('Wallet not found', 'WALLET_NOT_FOUND', 404);
    }
    
    $total_balance = $wallet['main_balance'] + $wallet['bonus_balance'];
    if ($total_balance < $stake_amount) {
        error('Insufficient balance', 'INSUFFICIENT_BALANCE');
    }
    
    // Generate unique room code
    do {
        $room_code = strtoupper(substr(md5(uniqid()), 0, 8));
        $stmt = $conn->prepare("SELECT id FROM rooms WHERE room_code = ?");
        $stmt->execute([$room_code]);
    } while ($stmt->fetch());
    
    // Create room
    $conn->beginTransaction();
    
    try {
        $stmt = $conn->prepare("
            INSERT INTO rooms (room_code, creator_id, stake_amount, status, expires_at) 
            VALUES (?, ?, ?, 'waiting', DATE_ADD(NOW(), INTERVAL 1 HOUR))
        ");
        $stmt->execute([$room_code, $user_id, $stake_amount]);
        $room_id = $conn->lastInsertId();
        
        $conn->commit();
        
        success([
            'room' => [
                'id' => $room_id,
                'room_code' => $room_code,
                'stake_amount' => $stake_amount,
                'status' => 'waiting',
                'expires_at' => date('Y-m-d H:i:s', strtotime('+1 hour'))
            ]
        ], 'Room created successfully');
        
    } catch (Exception $e) {
        $conn->rollBack();
        throw $e;
    }
    
} catch (Exception $e) {
    error_log("Create room error: " . $e->getMessage());
    error('Failed to create room', 'ROOM_ERROR', 500);
}
?>
