<?php
/**
 * Application Configuration
 */

// Load environment variables
if (file_exists(__DIR__ . '/../.env')) {
    $lines = file(__DIR__ . '/../.env', FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($lines as $line) {
        if (strpos($line, '=') !== false && strpos($line, '#') !== 0) {
            list($key, $value) = explode('=', $line, 2);
            $_ENV[trim($key)] = trim($value);
        }
    }
}

// CORS headers for API
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');
header('Content-Type: application/json; charset=utf-8');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Error reporting (disable in production)
if ($_ENV['APP_ENV'] !== 'production') {
    error_reporting(E_ALL);
    ini_set('display_errors', 1);
} else {
    error_reporting(0);
    ini_set('display_errors', 0);
}

// Timezone
date_default_timezone_set($_ENV['TIMEZONE'] ?? 'Africa/Dar_es_Salaam');

// Application constants
define('APP_NAME', 'Checkers Game');
define('APP_VERSION', '1.0.0');
define('JWT_SECRET', $_ENV['JWT_SECRET'] ?? 'your-secret-key');
define('JWT_ALGORITHM', 'HS256');
define('JWT_EXPIRY', 86400); // 24 hours

// Flutterwave configuration
define('FLUTTERWAVE_PUBLIC_KEY', $_ENV['FLUTTERWAVE_PUBLIC_KEY'] ?? '');
define('FLUTTERWAVE_SECRET_KEY', $_ENV['FLUTTERWAVE_SECRET_KEY'] ?? '');
define('FLUTTERWAVE_ENCRYPTION_KEY', $_ENV['FLUTTERWAVE_ENCRYPTION_KEY'] ?? '');

// Site configuration
define('SITE_URL', $_ENV['SITE_URL'] ?? 'https://yourdomain.com');
define('API_URL', SITE_URL . '/api');

// Game configuration
define('BOT_WINRATE', $_ENV['BOT_WINRATE'] ?? 60);
define('PLATFORM_FEE', $_ENV['PLATFORM_FEE'] ?? 5);
define('REFERRAL_BONUS', $_ENV['REFERRAL_BONUS'] ?? 50);

// Financial limits
define('MIN_DEPOSIT', $_ENV['MIN_DEPOSIT'] ?? 1000);
define('MIN_WITHDRAWAL', $_ENV['MIN_WITHDRAWAL'] ?? 5000);
define('MAX_WITHDRAWAL', $_ENV['MAX_WITHDRAWAL'] ?? 100000);

// Utility functions
function response($data, $status = 200) {
    http_response_code($status);
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit();
}

function success($data = null, $message = 'Success') {
    response([
        'success' => true,
        'message' => $message,
        'data' => $data
    ]);
}

function error($message = 'An error occurred', $code = 'ERROR', $status = 400) {
    response([
        'error' => true,
        'code' => $code,
        'message' => $message
    ], $status);
}

function validateRequired($data, $fields) {
    $missing = [];
    foreach ($fields as $field) {
        if (!isset($data[$field]) || empty($data[$field])) {
            $missing[] = $field;
        }
    }
    if (!empty($missing)) {
        error('Missing required fields: ' . implode(', ', $missing), 'MISSING_FIELDS');
    }
}

function sanitizeInput($data) {
    if (is_array($data)) {
        return array_map('sanitizeInput', $data);
    }
    return htmlspecialchars(strip_tags(trim($data)), ENT_QUOTES, 'UTF-8');
}

function generateToken($length = 32) {
    return bin2hex(random_bytes($length));
}

function generateReferralCode() {
    return 'REF' . str_pad(rand(1, 999999), 6, '0', STR_PAD_LEFT);
}

function hashPassword($password) {
    return password_hash($password, PASSWORD_DEFAULT);
}

function verifyPassword($password, $hash) {
    return password_verify($password, $hash);
}
?>
